%% Fitting a transition model to the data: hominin enamel stable C isotopes
%
% by Martin H. Trauth, 4 December 2020
% http://mres.uni-potsdam.de

%%
% Wynn, J., Alemseged, Z., Bobe, R., Grine, F., Negash, E., Sponheimer, M.
% (2020) Isotopic evidence for the timing of the dietary shift toward C4
% foods in eastern African Paranthropus. Proceedings of the National
% Academy of Sciences, 117, 21978–21984.
% https://dx.doi.org/10.1073/pnas.2006221117.
clear, clc, close all
dataraw = load('data_wynn_2020.txt');
data.Age = -dataraw(:,1);
data.d13C = dataraw(:,2);

%%
% Nonlinear least squares fit of a sigmoid function
fo = fitoptions('Method','NonlinearLeastSquares',...
    'StartPoint',[0 0 -2.8 1],...
    'Lower',[-20 -10 -5  0],...
    'Upper',[ 20  10  5 10]);
ft = fittype('a+b*sigmoidmht(x,c,d)',...
    'Coeff',{'a','b','c','d'},...
    'Options',fo)

%%
t = -3.5 : 0.1 : -1;
[f,fgof,fout] = fit(data.Age,data.d13C,ft)
y = f.a + f.b*sigmoidmht(t,f.c,f.d);

%%
rft = load('rampfit.dat');
rs = 'c = -2.411 (-2.467,-2.355)';

%%
xl = ['Age (Ma)'];
yl = ['\delta^{13}C_{enamel} (',char(8240),')'];

%%
figure('Position',[50 800 800 250])
axes('Position',[0.1 0.2 0.8 0.7],...
    'XLim',[-4 -1],...
    'YLim',[-20 5],...
    'FontSize',12,...
    'Box','On'), hold on
p = plot(f,data.Age,data.d13C);
p(1).Color = [0.1 0.3 0.6];
p(1).Marker = 'Square';
p(1).MarkerSize = 6;
p(1).MarkerFaceColor = [0.1 0.3 0.6];
p(1).MarkerEdgeColor = [0.1 0.3 0.6];
p(2).Color = [0.7 0.3 0.1];
p(2).LineWidth = 1.5;
p(2).LineStyle = '-';

line(-rft(:,1),rft(:,3),...                % Ramp Fit
    'Color',[0.7 0.3 0.1],...
    'LineWidth',1.5,...
    'LineStyle','-.')

lgd = legend('Data Points','Sigmoid Fit','Ramp Fit');
lgd.Location = 'southeast';
lgd.Box = 'Off';
lgd.FontSize = 12;
lgd.AutoUpdate = 'Off';
xlabel(xl,...
    'FontSize',12)
ylabel(yl,...
    'FontSize',12)

line([-2.41 -2.41],[-5 0],...              % Inflection Sigmoid Fit
    'LineWidth',1.5,...
    'Color',[0.7 0.3 0.1])
text(-2.41,2,'-2.41 Ma',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Right')

line([-2.41 -2.41],[-15 -10],...              % Inflection Ramp Fit
    'LineWidth',1.5,...
    'LineStyle','-.',...
    'Color',[0.7 0.3 0.1])
text(-2.41,-17,'-2.41 Ma',...
    'FontSize',18,...
    'Color',[0.7 0.3 0.1],...
    'HorizontalAlignment','Left')

print -dpng -r300 commands_wynn2020_sigmoid_ramp_vs2.png




